/***************************************************************** 
*  @file SST_HOST_TST_Message_Init.c
*  @brief Functions used for HOST-SEP communication to send message from HOST to SEP
*         that contains requested API to be executed in SEP and API parameters
*         Used for SEP External application ATP testing.
*         General HOST-SEP communication function for external ATP functions
*
*  @author TaniaM
*  @remarks Copyright (C) 2004 by Discretix Technologies Ltd.
*           All Rights reserved
*/

/********************** Include Files *************************/



#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "DX_VOS_Sem.h"
#include "SEPDriver.h" 
#include "ExtApp_qa_op_code.h"
#include "gen.h"
#include "error.h"
#include "sst_types.h"
#include "sst_errors.h"
#include "sst_sep_errors.h"
#include "ExtApp_HOST_TST_Message_Init.h"
#include <stdio.h>

/************************ Defines ******************************/
/************************ MACROS *******************************/
/************************ Global Data **************************/
/**************** Private function prototype *******************/
/************************ Public Functions *********************/


/********************************************************************************
* Function Name:                                                           
*  HOSTQA_ATP_Message_Init         
* 
* Inputs:
*  Host_opCode[in]           - OpCode of external application function on the SEP.
*  testData[in/out]          - pointer to data.
*
* Outputs:
*  DxUint32 - On success DX_OK is returned                                       
*                                                                 
* 
* Description:                                                                 
*  This function is used to operate the qa external application on the SEP, 
*  that executes appropriate ATP test according to Host OPCODE and update test data struct.
*
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with data 
* 3. write the message to shared memory                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*/
/******************************************************************************/   

IMPORT_C DxUint32 HOSTQA_ATP_Message_Init(DxUint32_t Host_opCode,Ext_ATP_DataOpTestDataStruct* testData)
{

   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;

  
   /* The return error identifier */
   DxUint32_t Host_error = DX_OK;  

   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams=0;
   
   DxUint32_t   Host_maxLength;
  
  
   /* parameters */
   DxUint32_t*   Host_messageParam = DX_NULL;
   
   
   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);

   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   if (testData != DX_NULL)
   {
  	  /* 2. prepare message with data  */
  	  /*********************************/ 
  	  /* convert to the words, because Host_messageParam is array of words */
  	  Host_numOfSimpleParams = sizeof(Ext_ATP_DataOpTestDataStruct)/4;
  	  
  	  Host_messageParam =(DxUint32_t*)DX_VOS_MemMalloc (sizeof(Ext_ATP_DataOpTestDataStruct));
  	  
	  if (Host_messageParam == DX_NULL)
  	      Host_error = DX_MEM_ALLOCATION_ERROR;
  	  
  	  if(Host_error != DX_OK)
       {
            goto end_function_unlock;
       }
  	  
       DX_VOS_FastMemCpy(Host_messageParam,testData,sizeof(Ext_ATP_DataOpTestDataStruct));

       /* 3. write message to shared memory */
       /*************************************/
       Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                           sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                           sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                           &Host_sramOffset,
                                           DX_FALSE);
                                           
                                        

     if(Host_error != DX_OK)
     {
          goto end_function_unlock;
     }
 
   }

   
 
  
   /* 4. end message */
   /******************/
   SEPDriver_EndMessage(Host_sramOffset);
           

   /* 5. wait for the response of SEP side */
   /****************************************/
   Host_error = SEPDriver_POLL_FOR_REPONSE();

   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }


   /* 6. start reading message from the SEP  and check returned function status from SEP */
   /****************************************************************************/
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);

   if(Host_error != DX_OK)
   {
      goto end_function_unlock;
   }


       Host_maxLength = ((sizeof(Ext_ATP_DataOpTestDataStruct) + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t); 
       
       /* read  the updated data */
       SEPDriver_ReadParamater((DxUint32_t)testData,
                                sizeof(Ext_ATP_DataOpTestDataStruct),
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
                                
                                
      
end_function_unlock:   
 

   /* unlock access to the SEP */

   SEPDriver_Unlock();

   return Host_error;

   

}/* END OF HOSTQA_ATP_Message_Init */
